<?php

namespace App\Http\Controllers;

use App\Http\Requests\MembershipRequest;
use App\Models\MembershipType;
use App\Models\Membership;
use App\Models\Member;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;

class MembershipController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->has('search')) {
            $keyword = $request->input('search');
            $paginate = false;

            $memberships = Membership::whereHas('member', function ($query) use ($keyword) {
                $query->where('name', 'LIKE', '%' . $keyword . '%')
                    ->orWhere('reg_no', 'LIKE', '%' . $keyword . '%')
                    ->orWhere('phone', 'LIKE', '%' . $keyword . '%');
            })->with(['member', 'membership_type', 'transactions'])->orderBy('created_at', 'DESC')->get();

            return view('membership.list', compact('memberships', 'paginate'));
        } else {
            $paginate = true;
            $memberships = Membership::with('member', 'membership_type', 'transactions')->orderBy('created_at','DESC')->paginate(10)->onEachSide(0);

            return view('membership.list', compact('memberships', 'paginate'));
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create($id = 0)
    {
        $types = MembershipType::get();
        $members = Member::where('status', 'active')->get();
        return view('membership.create', compact('types', 'members', 'id'));
    }   

    /**
     * Store a newly created resource in storage.
     */
    public function store(MembershipRequest $request)
    {
        $member_id = $request->input('member');
        $membership_type_id = $request->input('type');
        $membership_type = MembershipType::find($membership_type_id);

        $member = Member::find($member_id);

        $membership = $member->memberships()->create([
            'membership_type_id' => $membership_type->id,
            'valid_from' => $request->input('valid_from'),
            'valid_to' => $request->input('valid_to')
        ]);

        $membership->transactions()->create([
            'amount' => $membership_type->amount,
            'fee' => 0,
            'dr_cr' => 'dr',
            'date' => Carbon::now(),
            'method' => 'online',
            'note' => $membership_type->name . ' Membership fee',
            'created_user_id' => Auth::user()->id,
            'type' => 'membership fee'
        ]);

        return back()->with('success', 'Saved successfully');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Membership $membership)
    {
        $transactions = $membership->transactions;
        foreach ($transactions as $transaction) {
            Transaction::where('id', $transaction->id)->delete();
        }
        $membership->delete();
        return back()->with('success', 'Deleted successfully');
    }
}